
/*******************************************************************************
		DATASET CREATOR - INTERGENERATIONAL TRANSFERS IN INFANT MORTALITY

		
Program which ceates a dataset for analysing intergenerational transfers in infant 
mortality along the maternal line using databases that are stored in the IDS	
					
			
			
Version: 1
Date released: 2018/05/08
Author and contact: Luciana Quaranta, luciana.quaranta@ekh.lu.se
	
	
Tables and variables used 

-INDIVIDUAL	
		- Type Birth_date / Baptism_date(with specification of Date_type)
		- Type Death_date / Burial_date 
		- Type Sex
		- Type Start_observation
		- Type End_observation
		- Type Marriage_date
	
-INDIV_INDIV
		- Relation Mother
		- Relation Father
		- Relation Husband
	
*******************************************************************************/





/*******************************************************************************
						PART 1: DATA CONTROLS
*******************************************************************************/




/*Change working directory, specifying the folder containing the IDS tables. E.g.: 
cd "c:\Users\Luciana\IDS\InfantMort"
*/

discard
/*Set PERSONAL directory for ado files (folder where IDS programs are saved). E.g.:

sysdir set PERSONAL "c:\Users\Luciana\IDS\InfantMort"
*/
clear

/*Check for problems in the INDIVIDUAL and INDIV_INDIV tables*/

use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Type=="birth_date" | Type=="death_date" | Type=="baptism" | Type=="baptism_date" | Type=="burial" | Type=="burial_date" | Type=="sex"
gen Date=mdy(Month,Day,Year)
format Date %td
keep Id_I Type Date
sort Id_I Type Date
bysort Id_I Type (Date): gen totrows = _N
tab Type if totrows>1
if r(N)==0{
display "No duplicates found in types birth_date, death_date, baptism, baptism_date, burial, burial_date or sex in the INDIVIDUAL table. You can continue running the program."
}
if r(N)>0{
display as error "Duplicates found in types birth_date, death_date, baptism, baptism_date, burial, burial_date or sex. Individuals with problems are shown in the data browse. Fix the errors and restart the program "
browse Id_I Type Date if totrows>1
}


use INDIV_INDIV.dta, clear
replace Relation = lower(Relation)
keep if Relation=="mother" | Relation=="father"
keep Id_I_1 Id_I_2 Relation
bysort Id_I_2 Relation: gen totrows=_N
sort Id_I_2 Id_I_1
tab Relation if totrows>1
if r(N)==0{
display "There are no children linked to more than one mother or father in the INDIV_INDIV table. You can continue running the program."
}
if r(N)>0{
display as error "There are children linked to more than one mother or father in the INDIV_INDIV table. Individuals with problems are shown in the data browse. Fix the errors and restart the program "
browse Id_I_1 Id_I_2 Relation if totrows>1
}






/*******************************************************************************
						PART 2: BASIC INFORMATION
*******************************************************************************/

*Birth date
use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Type=="birth_date"
gen BirthDate=mdy(Month,Day,Year)
capture gen start_date=mdy(Start_month, Start_day, Start_year) if Date_type=="Declared" & Estimation=="Age_based"
capture gen end_date=mdy(End_month, End_day, End_year) if Date_type=="Declared" & Estimation=="Age_based"
capture replace  BirthDate=start_date+((end_date-start_date)/2) if (end_date-start_date)<367 & end_date!=.
format BirthDate %td
capture drop start_date end_date
keep Id_I BirthDate
gen firstBday=mdy(month(BirthDate),day(BirthDate),year(BirthDate)+1)
replace firstBday=mdy(2,28,year(BirthDate)+1) if firstBday==. & day(BirthDate)==29 & month(BirthDate)==2
format firstBday %td
save BirthDate.dta, replace

*Birth event
use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Type=="birth_date" 
keep if Date_type=="Event"
capture gen Birth_event=mdy(Month,Day,Year)
capture format Birth_event %td
capture keep Id_I Birth_event
drop if Birth_event==.
save Birth_event.dta, replace

*Baptism event
use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Type=="baptism" | Type=="baptism_date"
capture keep if Date_type=="Event"
capture gen Baptism_event=mdy(Month,Day,Year)
capture format Baptism_event %td
capture keep Id_I Baptism_event
drop if Baptism_event==.
save Baptism_event.dta, replace

*Birth or baptism event
use Birth_event.dta,clear
merge 1:1 Id_I using Baptism_event.dta
drop _merge
gen Birth_baptism_event=Birth_event
replace Birth_baptism_event=Baptism_event if Birth_event==.
format Birth_baptism_event %td
drop if Birth_baptism_event==.
keep Id_I Birth_baptism_event
save Birth_baptism_event.dta, replace

*Death date
use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Type=="death_date" 
gen DeathDate=mdy(Month,Day,Year)
format DeathDate %td
keep Id_I DeathDate
drop if DeathDate==.
save DeathDate.dta, replace


*Husband death date
use INDIV_INDIV.dta, clear
keep if Relation=="Husband"
keep Id_I_1 Id_I_2
duplicates drop
rename Id_I_1 Id_I
capture merge m:1 Id_I using DeathDate.dta
capture keep if _merge==3
capture drop _merge
capture rename DeathDate HusbandDeathDate
capture bysort Id_I_2: gen totHusb=_N
capture keep if totHusb==1
capture keep Id_I_2 HusbandDeathDate
capture rename Id_I_2 Id_I
drop if HusbandDeathDate==.
save HusbandDeathDate.dta, replace


*Sex
use INDIVIDUAL.dta, clear
replace Type = lower(Type)
replace Value = lower(Value)
keep if Type=="sex" 
rename Value Sex
keep Id_I Sex
drop if Sex==""
save Sex.dta, replace

*Start observation (stored in INDIVIDUAL or created from events)
use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Type=="start_observation" 
gen FirstDate=mdy(Month,Day,Year)
format FirstDate %td
keep Id_I FirstDate
drop if FirstDate==.
save FirstDate.dta, replace

use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Date_type=="Event"  
gen FirstDate=mdy(Month,Day,Year)
format FirstDate %td
keep Id_I FirstDate
append using FirstDate.dta
drop if FirstDate==.
*Keep first start observation date
bysort Id_I (FirstDate): gen row = _n
drop if row>1
keep Id_I FirstDate
save FirstDate.dta, replace



*End observation (stored in INDIVIDUAL or created from events)
use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Type=="end_observation" 
gen LastDate=mdy(Month,Day,Year)
format LastDate %td
keep Id_I LastDate
drop if LastDate==.
save LastDate.dta, replace

use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Date_type=="Event"  
gen LastDate=mdy(Month,Day,Year)
format LastDate %td
keep Id_I LastDate
append using LastDate.dta
drop if LastDate==.
bysort Id_I (LastDate): gen lastrow = (_n==_N)
keep if lastrow==1
keep Id_I LastDate
save LastDate.dta, replace

*First marriage date
use INDIVIDUAL.dta, clear
replace Type = lower(Type)
keep if Type=="marriage_date" 
gen FirstMarriageDate=mdy(Month,Day,Year)
format FirstMarriageDate %td
drop if FirstMarriageDate==.
*Keep first marriage date
capture bysort Id_I (FirstMarriageDate): gen row = _n
capture keep if row==1
keep Id_I FirstMarriageDate
save FirstMarriageDate.dta, replace


*Last child birth date
use INDIV_INDIV.dta, clear
replace Relation=lower(Relation)
keep if Relation=="mother" 
keep Id_I_1 Id_I_2
rename Id_I_1 MotherID
rename Id_I_2 Id_I
merge 1:1 Id_I using Birth_baptism_event.dta
keep if _merge==3
keep MotherID Birth_baptism_event
drop if Birth_baptism_event==.
rename MotherID Id_I
collapse (max) LastBirthEvent= Birth_baptism_event,by(Id_I)
save LastBirthEvent.dta, replace

*Last child birth date for the father
use INDIV_INDIV.dta, clear
replace Relation=lower(Relation)
keep if Relation=="father" 
keep Id_I_1 Id_I_2
rename Id_I_1 FatherID
rename Id_I_2 Id_I
merge 1:1 Id_I using Birth_baptism_event.dta
keep if _merge==3
keep FatherID Birth_baptism_event
rename FatherID Id_I
collapse (max) LastBirthEvent= Birth_baptism_event,by(Id_I)
append using LastBirthEvent.dta
save LastBirthEvent.dta, replace


use BirthDate.dta, clear
merge 1:1 Id_I using Birth_event.dta
drop if _merge==2
drop _merge
merge 1:1 Id_I using Baptism_event.dta
drop if _merge==2
drop _merge
merge 1:1 Id_I using DeathDate.dta
drop if _merge==2
drop _merge
merge 1:1 Id_I using Sex.dta
drop if _merge==2
drop _merge
merge 1:1 Id_I using FirstDate.dta
drop if _merge==2
drop _merge
merge 1:1 Id_I using LastDate.dta
drop if _merge==2
drop _merge
merge 1:1 Id_I using HusbandDeathDate.dta
drop if _merge==2
drop _merge
merge 1:1 Id_I using FirstMarriageDate.dta
drop if _merge==2
drop _merge
merge 1:1 Id_I using LastBirthEvent.dta
drop if _merge==2
drop _merge


replace Baptism_event=. if Birth_event!=. & Baptism_event!=.

/*Individuals for whom a baptism event is observed are assumed to be under 
observation from birth*/
replace FirstDate=BirthDate if Baptism_event==FirstDate & Baptism_event!=.

/*Replace last date with last birth date if last birth date occurs later*/
replace LastDate=LastBirthEvent if LastBirthEvent>LastDate & LastBirthEvent!=.

/*Replace last date with death date if last date occurs after death date- 
If funeral dates were observed, they occur after the death date. By this step 
the last date of observation is corrected to the death date instead of the funeral 
date. 
*/
replace LastDate=DeathDate if LastDate>DeathDate & DeathDate!=.

save Basic_information.dta, replace

*erase BirthDate.dta
erase Birth_event.dta
erase Baptism_event.dta
erase Sex.dta
erase FirstDate.dta
erase LastDate.dta
erase FirstMarriageDate.dta
erase LastBirthEvent.dta
erase Birth_baptism_event.dta


/*******************************************************************************
						PART 3: WOMEN AND BIRTHS INFORMATION
*******************************************************************************/



***** Child mother link
use INDIV_INDIV.dta, clear
replace Relation = lower(Relation)
keep if Relation=="mother" 
keep Id_I_1 Id_I_2
rename Id_I_1 MotherID
rename Id_I_2 Id_I

merge 1:1 Id_I using Basic_information.dta
drop if _merge==2
drop _merge
drop FirstDate FirstMarriageDate

gen unknownBdate=(BirthDate==.)


*Adjustments of birthdates for twins born one day apart (date of first birth assigned)
sort MotherID BirthDate
gen twin1daydiff =(MotherID==MotherID[_n-1] & BirthDate==BirthDate[_n-1]+1)
replace BirthDate=BirthDate[_n-1] if twin1daydiff==1 & MotherID==MotherID[_n-1]
replace firstBday=firstBday[_n-1] if twin1daydiff==1 & MotherID==MotherID[_n-1]
replace Birth_event=Birth_event[_n-1] if twin1daydiff==1 & MotherID==MotherID[_n-1]
drop twin1daydiff


gen InfantDeath=(DeathDate<firstBday & DeathDate!=.)

foreach Sexvar in female male{
gen `Sexvar'_Birth=(Sex=="`Sexvar'")
gen `Sexvar'_InfantDeath = (InfantDeath==1 & Sex=="`Sexvar'")
}

/*Adjustment of sex to multiple for twins of different sex. 
Twins are dropped at a later stage when creating mother specific variables*/
bysort MotherID BirthDate: gen nbirths=_N
replace Sex="Multiple" if nbirths>1 & MotherID==MotherID[_n+1] & BirthDate==BirthDate[_n+1] & Sex!=Sex[_n+1]
replace Sex="Multiple" if nbirths>1 & MotherID==MotherID[_n-1] & BirthDate==BirthDate[_n-1] & Sex!=Sex[_n-1]


bysort MotherID (BirthDate): gen border=_n

collapse (min) ChildID=Id_I (max) DeathDate=DeathDate (max) LastChildDate=LastDate (max) border=border (sum) Births_unknownBdate=unknownBdate (sum) Female_Birth=female_Birth (sum) Male_Birth=male_Birth (sum) Female_InfantDeath=female_InfantDeath (sum) Male_InfantDeath=male_InfantDeath (sum) InfantDeath=InfantDeath,by(MotherID BirthDate Birth_event Baptism_event nbirths Sex firstBday)
/*The maximum death date and birth order is selected for twins. 
Twins are dropped later on.*/

replace LastChildDate=DeathDate if DeathDate!=. & LastChildDate>DeathDate

gen missing=(BirthDate==.)
bysort MotherID: egen missingbdate=max(missing)
drop missing
save Births.dta, replace





*Woman observed for entire reproductive period
use Births.dta,clear
bysort MotherID (BirthDate): gen row = _n
keep if row==1
keep MotherID BirthDate 
rename BirthDate first_child_bdate
rename MotherID Id_I
save first_child_bdate.dta, replace

use Births.dta,clear
keep MotherID
duplicates drop
rename MotherID Id_I
merge m:1 Id_I using Basic_information.dta
drop if _merge==2
drop _merge
drop Sex 

merge m:1 Id_I using first_child_bdate.dta
drop if _merge==2
drop _merge

foreach bd in 20 50{
gen Bday`bd'=mdy(month(BirthDate),day(BirthDate),year(BirthDate)+`bd')
replace Bday`bd'=mdy(2,28,year(BirthDate)+1) if Bday`bd'==. & day(BirthDate)==29 & month(BirthDate)==2
format Bday`bd' %td
}

gen FromStart=(FirstDate<=Bday20 & FirstDate!=.)
replace FromStart = 1 if FirstDate<=FirstMarriageDate & FirstMarriageDate!=. & FirstDate!=. & ((FirstMarriageDate<first_child_bdate & FirstMarriageDate!=.)| first_child_bdate==.)
gen ToEnd=(LastDate>=Bday50 & LastDate!=.)
gen ToEnd_death=((LastDate>=Bday50 & LastDate!=.)| LastDate==DeathDate)
gen ToEnd_hb=(LastDate>=Bday50 & HusbandDeathDate>=Bday50 & LastDate!=.)

gen Obs_2050=(FromStart==1 & ToEnd==1)
gen Obs_2050death=(FromStart==1 & ToEnd_death==1)
gen Obs_2050hb=(FromStart==1 & ToEnd_hb==1)

drop Bday20 Bday50 FromStart ToEnd ToEnd_death ToEnd_hb

save Women.dta, replace






/*******************************************************************************
						PART 4: VARIABLES FOR ANALYSIS
*******************************************************************************/

***** Child sex
use Births.dta, clear
drop if missingbdate==1
keep ChildID BirthDate Sex
drop BirthDate 
gen Type="Child_sex"
rename Sex Value
save ChildSex.dta, replace


***** Birth order
use Births.dta, clear
drop if missingbdate==1
keep ChildID border BirthDate 
gen Type="Birth_order"
drop BirthDate 
tostring border,gen(Value)
drop border 
save birthorder.dta, replace


***** Multiple births
use Births.dta, clear
drop if missingbdate==1
keep ChildID nbirths BirthDate 
drop BirthDate 
gen Type="N_births"
tostring nbirths, gen(Value)
drop nbirths
save multiplebirth.dta, replace


***** Child infant death event
use Births.dta, clear
drop if missingbdate==1
keep if DeathDate<firstBday
gen DayFrac=0.02 if BirthDate==DeathDate /*Adjustment of dates for children who are born and die on the same day*/
keep ChildID DeathDate DayFrac
gen Year=year(DeathDate)
gen Month=month(DeathDate)
gen Day=day(DeathDate)
drop DeathDate
gen Type="Infant_death"
gen Value="1"
save infantdeath.dta, replace


***** Child birth date
use Births.dta, clear
drop if missingbdate==1
keep ChildID BirthDate
drop if BirthDate==.
gen Year=year(BirthDate)
gen Month=month(BirthDate)
gen Day=day(BirthDate)
drop BirthDate 
gen Type="Child_birthdate"
save childBirthdate.dta, replace


***** Mother's birthdate
use Births.dta, clear
drop if missingbdate==1
rename MotherID Id_I
keep Id_I ChildID
duplicates drop
merge m:1 Id_I using Basic_information.dta
keep if _merge==3
drop _merge
keep ChildID BirthDate
drop if BirthDate==.
gen Year=year(BirthDate)
gen Month=month(BirthDate)
gen Day=day(BirthDate)
drop BirthDate 
gen Type="Mother_birthdate"
save motherBirthdate.dta, replace

***** Mother ID
use Births.dta, clear
drop if missingbdate==1
keep MotherID ChildID
gen Type="MotherID"
tostring MotherID, gen(Value)
drop MotherID
save motherid.dta, replace






/*******************************************************************************
							AT RISK VARIABLE						

-The "at risk" variable is used by the episodes file creator. Only rows where 
the individuals is at risk of experiencing the event are kept. 
-Only children who have an observed birth event have values for the variable. 
-The value is set to 1 at the birth of the child if there is a birth event. 
-The value is set to 0 at the death date of the child or date or date of 
out-migration (if either occur below age 1)
-If the end date is the same as the start date (e.g. death on the date of birth), 
the DayFrac is set to 0.02 so that children become at risk before stopping being
at risk. 
*******************************************************************************/

*Start at birth event
use Births.dta, clear
drop if missingbdate==1
drop if Birth_event==. & Baptism_event==.
keep ChildID BirthDate
gen Year=year(BirthDate)
gen Month=month(BirthDate)
gen Day=day(BirthDate)
drop BirthDate
gen Type="At_risk"
gen Value="1"
save atrisk.dta, replace

*End on child's last date
use Births.dta, clear
drop if missingbdate==1
drop if Birth_event==. & Baptism_event==.
drop if LastChildDate==.
gen DayFrac=0.02 if BirthDate==LastChildDate
gen  EndRiskDate=LastChildDate
format EndRiskDate %td
replace EndRiskDate=firstBday if firstBday<EndRiskDate
replace EndRiskDate=firstBday if EndRiskDate==BirthDate & EndRiskDate!=DeathDate 
gen Year=year(EndRiskDate)
gen Month=month(EndRiskDate)
gen Day=day(EndRiskDate)
keep ChildID Day Month Year DayFrac
gen Type="At_risk"
gen Value="0"
append using atrisk.dta
sort ChildID Day Month Year DayFrac
save atrisk.dta, replace




/*******************************************************************************
							CHILD CENSORING						

-The variable "Under_observation" is created
	-The value 1 is assigned at birth (if a birth or a baptism event is observed)
	-The value 0 is assigned
		-at the date of death of the child (if < first birthday)
		-on the last date of observation of the mother or child (if < first birthday)
		-at the first birthday if either the mother or the child are observed 
		after the child's first birthday. 

-This variable is meant to be used as sensitivity analysis. After running a 
model where all children are considered under observation until their first 
birthday unless they die or out-migrate before they turn one, a model that 
censors children on the last date of observation of the mother/child can be 
estimated. The information used to define the last date of observation for the 
mother/child are own events, End_observation dates and dates of birth of 
children. This means that for family reconstitution data children who are the 
last born and whose mother does not die in the study area and who they 
themselves don't marry, have children or die in the study area will be censored 
on their date of birth. 

	
	
*******************************************************************************/

*Start at birth event
use Births.dta, clear
drop if missingbdate==1
drop if Birth_event==. & Baptism_event==.
keep ChildID BirthDate
gen Year=year(BirthDate)
gen Month=month(BirthDate)
gen Day=day(BirthDate)
drop BirthDate
gen Type="Under_observation"
gen Value="1"
save ChildUnderObservation.dta, replace



*End on child's last date
use Births.dta, clear
drop if missingbdate==1
drop if Birth_event==. & Baptism_event==.
drop if LastChildDate==.
gen DayFrac=0.02 if BirthDate==LastChildDate
keep ChildID LastChildDate DayFrac BirthDate DeathDate
rename LastChildDate EndRiskDate
save End_risk.dta, replace

*End on woman's exit
use Births.dta, clear
drop if missingbdate==1
drop if Birth_event==. & Baptism_event==.
rename MotherID Id_I 
rename BirthDate ChildBirthDate
rename firstBday firstBdayChild
rename DeathDate DeathDateChild
keep Id_I ChildID ChildBirthDate firstBdayChild DeathDateChild LastChildDate
duplicates drop
merge m:1 Id_I using Basic_information.dta
keep if _merge==3
gen DayFrac=0.02 if ChildBirthDate==LastDate
drop if LastDate==.
drop if LastDate>DeathDateChild & DeathDateChild!=.
keep if LastDate>LastChildDate
keep ChildID LastDate DayFrac ChildBirthDate DeathDateChild firstBdayChild
rename ChildBirthDate BirthDate
rename DeathDateChild DeathDate
rename LastDate EndRiskDate
rename firstBdayChild firstBday
append using End_risk.dta
duplicates drop
sort ChildID EndRiskDate DayFrac
by ChildID: gen keepr=1 if _n==_N
keep if keepr==1
drop keepr
replace EndRiskDate=firstBday if firstBday<EndRiskDate
save End_risk.dta, replace

*
use End_risk.dta,clear
gen Year=year(EndRiskDate)
gen Month=month(EndRiskDate)
gen Day=day(EndRiskDate)
drop EndRiskDate
gen Type="Under_observation"
gen Value="0"
append using ChildUnderObservation.dta
gsort ChildID Year Month Day -DayFrac
keep ChildID Year Month Day DayFrac Type Value
save ChildUnderObservation.dta, replace

erase End_risk.dta







/*******************************************************************************
					INTERGENERATIONAL TRANSMISSION VARIABLES
*******************************************************************************/

***** Define grandmothers
use INDIV_INDIV.dta, clear
keep if Relation=="Mother" | Relation=="MOTHER"
keep Id_I_1 Id_I_2
rename Id_I_1 GrandMotherID
rename Id_I_2 MotherID
merge 1:m MotherID using Births.dta
drop if missingbdate==1
keep if _merge==3
drop _merge
keep ChildID GrandMotherID
save Mothers_G.dta, replace


***** Grandmother's ID
use Mothers_G.dta, clear
keep ChildID GrandMotherID
gen Type="GrandmotherID"
tostring GrandMotherID, gen(Value)
drop GrandMotherID
save grandmotherID.dta, replace


***** Grandmother's birthdate
use Mothers_G.dta, clear
keep ChildID GrandMotherID
rename GrandMotherID Id_I
merge m:1 Id_I using Basic_information.dta
keep if _merge==3
drop _merge
keep ChildID BirthDate
drop if BirthDate==.
gen Year=year(BirthDate)
gen Month=month(BirthDate)
gen Day=day(BirthDate)
drop BirthDate
gen Type="Grandmother_birthdate"
save grandmotherbdate.dta, replace


**** Intergenerational variable (number of births and infant deaths. Twins are counted)
use Births.dta,clear
rename Births_unknownBdate G0_Births_unknownBdate
keep MotherID Female_Birth Male_Birth Female_InfantDeath Male_InfantDeath InfantDeath nbirths BirthDate G0_Births_unknownBdate
gen multiple=(nbirths>1 & BirthDate!=.)
collapse (sum) G0_mult_b = multiple (sum) G0_Births=nbirths (sum) G0_Births_unknownBdate=G0_Births_unknownBdate (sum) G0_InfD=InfantDeath (sum) G0_F_InfD=Female_InfantDeath (sum) G0_M_InfD=Male_InfantDeath (sum) G0_M_Births=Male_Birth (sum) G0_F_Births=Female_Birth,by(MotherID)
rename MotherID GrandMotherID
merge 1:m GrandMotherID using Mothers_G.dta
keep if _merge==3
drop _merge
duplicates drop
drop GrandMotherID
save intergenerational.dta, replace

capture erase intergenerational_l.dta
global vname "G0_mult_b G0_Births G0_InfD G0_F_InfD G0_M_InfD G0_M_Births G0_F_Births G0_Births_unknownBdate"
foreach lname of global vname{
use intergenerational.dta, clear
keep ChildID `lname'
gen Type="`lname'"
rename `lname' Value
capture append using intergenerational_l.dta
save intergenerational_l.dta, replace
}

use intergenerational_l.dta, clear
rename Value temp
tostring temp, gen(Value)
drop temp
save intergenerational_l.dta, replace




**** Grandmother below age 50
use Mothers_G.dta, clear
rename GrandMotherID Id_I
merge m:1 Id_I using Basic_information.dta 
drop if _merge==2
drop _merge
keep ChildID BirthDate
rename BirthDate GrandmothersBdate
merge 1:1 ChildID using Births.dta
drop if missingbdate==1
drop if _merge==2
drop _merge
keep if Birth_event!=.
keep ChildID GrandmothersBdate BirthDate firstBday DeathDate
gen gage=(BirthDate-GrandmothersBdate)/365.25
gen Value=(gage<50)
replace Value=-1 if GrandmothersBdate==.
replace Value=-1 if gage<30
gen bday50g0=mdy(month(GrandmothersBdate),day(GrandmothersBdate),year(GrandmothersBdate)+50)
replace bday50g0=mdy(2,28,year(GrandmothersBdate)+50) if bday50g0==. & day(BirthDate)==29 & month(BirthDate)==2
format bday50g0 %td
replace Value=0 if bday50g0==BirthDate
save temp.dta, replace

use temp.dta,clear
gen Day=day(BirthDate)
gen Month=month(BirthDate)
gen Year=year(BirthDate)
gen Type="GO_under_age_50"
keep ChildID Day Month Year Type Value
save GO_under_age_50.dta, replace

use temp.dta,clear
keep if Value==1
keep if bday50g0< firstBday
drop if bday50g0>=DeathDate & DeathDate!=.
drop Value 
gen Value=0
gen Day=day(bday50g0)
gen Month=month(bday50g0)
gen Year=year(bday50g0)
gen Type="GO_under_age_50"
keep ChildID Day Month Year Type Value
append using GO_under_age_50.dta
rename Value temp
tostring temp, gen(Value)
drop temp
save GO_under_age_50.dta, replace

erase temp.dta



**** Grandmother in sample

use Mothers_G.dta, clear
rename GrandMotherID Id_I
merge m:1 Id_I using Women.dta
drop if _merge==2
drop _merge
save temp.dta, replace

use temp.dta,clear
gen Type="G0_obs_2050"
tostring Obs_2050,gen(Value)
keep ChildID Type Value
save grandmother_complete.dta, replace

use temp.dta,clear
gen Type="G0_obs_2050death"
tostring Obs_2050death,gen(Value)
keep ChildID Type Value
append using grandmother_complete.dta
save grandmother_complete.dta, replace

use temp.dta,clear
gen Type="G0_obs_2050husband"
tostring Obs_2050hb,gen(Value)
keep ChildID Type Value
append using grandmother_complete.dta
save grandmother_complete.dta, replace

erase temp.dta




**** Mother in sample

use Births.dta,clear
drop if missingbdate==1
keep MotherID ChildID
rename MotherID Id_I
merge m:1 Id_I using Women.dta
drop if _merge==2
drop _merge
save temp.dta, replace

use temp.dta,clear
gen Type="G1_obs_2050"
tostring Obs_2050,gen(Value)
keep ChildID Type Value
save mother_complete.dta, replace

use temp.dta,clear
gen Type="G1_obs_2050death"
tostring Obs_2050death,gen(Value)
keep ChildID Type Value
append using grandmother_complete.dta
save mother_complete.dta, replace

use temp.dta,clear
gen Type="G1_obs_2050husband"
tostring Obs_2050hb,gen(Value)
keep ChildID Type Value
append using grandmother_complete.dta
save mother_complete.dta, replace

erase temp.dta






/*******************************************************************************
						PART 5: EPISODES FILE
*******************************************************************************/


**** Chronicle file
use ChildSex.dta, clear
append using birthorder.dta
append using multiplebirth.dta
append using infantdeath.dta
append using childBirthdate.dta
append using motherBirthdate.dta
append using motherid.dta
append using ChildUnderObservation.dta
append using atrisk.dta

append using grandmotherID.dta
append using intergenerational_l.dta
append using grandmother_complete.dta
append using grandmotherbdate.dta
append using GO_under_age_50

append using mother_complete.dta

rename ChildID Id_I
duplicates drop

order Id_I Type Value Year Month Day DayFrac

save Chronicle.dta, replace




**** Variable setup file
use Chronicle.dta, clear
gen Transition="Invariant" if Year==.
keep Type Transition
duplicates drop
replace Transition="End" if Type=="Infant_death"
replace Transition="Invariant" if Type=="Mother_birthdate"
replace Transition="Invariant" if Type=="Child_birthdate"
replace Transition="Invariant" if Type=="Grandmother_birthdate"
replace Transition="Start" if Transition==""
gen Duration="Continuous"
save VarSetup.dta, replace


EpisodesFileCreator , atrisk(At_risk)

local fn = "`c(filename)'"
capture erase "Episodes_file.dta"
shell rename "`fn'" "Episodes_file.dta"


/*The dataset can be saved into older versions of Stata with the command:
saveold Episode_file_v12 , version(12)
*/



/*******************************************************************************
					PART 6:  ERASE TEMPORARY FILES
*******************************************************************************/

capture erase Mothers_G.dta
capture erase ChildSex.dta
capture erase birthorder.dta
capture erase motherBirthdate.dta
capture erase grandmotherID.dta
capture erase grandmother_complete.dta
capture erase intergenerational.dta
capture erase intergenerational_l.dta
capture erase multiplebirth.dta
capture erase infantdeath.dta
capture erase atrisk.dta
capture erase Basic_information.dta
capture erase ChildBasicInfo.dta
capture erase childBirthdate.dta
capture erase motherid.dta
capture erase BirthDate.dta
capture erase Chronicle.dta
capture erase temp.dta
capture erase VarSetup.dta
capture erase grandmotherbdate.dta
capture erase first_child_bdate.dta
capture erase DeathDate.dta
capture erase HusbandDeathDate.dta
capture erase GO_under_age_50.dta
capture erase Births.dta
capture erase Women.dta
capture erase mother_complete.dta
capture erase ChildUnderObservation.dta





