
/*******************************************************************************
STATISTICAL ANALYSIS PROGRAM - INTERGENERATIONAL TRANSFERS IN INFANT MORTALITY

		
Program which runs the statistical analysis to study intergenerational transfers 
in infant mortality along the maternal line using databases that are stored in 
the IDS
			
			
			
Version: 1
Date released: 2018/05/08
Author and contact: Luciana Quaranta, luciana.quaranta@ekh.lu.se
	
*******************************************************************************/

*Change working directory



use Episodes_file.dta, clear

/* If the name of the episodes file was not changed at the end of the program 
to create the dataset, in the use command specify the file name that was given 
after running the episodes file creator

use Episodes_file_DATE_TIME.dta, clear

*/



/*******************************************************************************
			PART 1: CREATION OF ADDITIONAL VARIABLES 
*******************************************************************************/
rename Id_I ChildID

gen Child_birthyear=year(Child_birthdate)
gen Mother_birthyear=year(Mother_birthdate)
gen Grandmother_birthyear=year(Grandmother_birthdate)

bysort ChildID (date1 date2): gen firstrow=(_n==_N)
gen temp = Child_birthyear if firstrow==1
egen temp1=mean(temp)
global Mean_child_byear= round(temp1,1)
gen Child_birthdate_cent=(Child_birthdate-mdy(1,1,$Mean_child_byear ))/365.2425
drop temp temp1 firstrow
display "Child_birthdate_cent is centered on January 1st, " $Mean_child_byear as result

egen G0_InfD_cat=cut( G0_InfD),at(0 1 2 100)
tostring G0_InfD_cat, replace
replace G0_InfD_cat="2+" if G0_InfD_cat=="2"
char G0_InfD_cat [omit] "0"

gen Mother_age = (Child_birthdate-Mother_birthdate)/365.2425
egen Mother_age_cat=cut(Mother_age),at(15 25 35 50)
char Mother_age_cat [omit] 25

egen G0_Births_cat=cut(G0_Births),at(2 3 4 7 100)
char G0_Births_cat [omit] 2

egen Birth_order_cat=cut(Birth_order),at(1 2 3 4 7 100)
char Birth_order_cat [omit] 1


/*******************************************************************************
				PART 2: SELECTION OF THE STUDY SAMPLE
*******************************************************************************/

*Keeping only cases where the grandmother ID is known 
format GrandmotherID %12.0f
format MotherID %12.0f
drop if GrandmotherID==-1

*Tabulating multiple births (for the mother) and keeping only singleton births
tab N_births
keep if N_births==1
drop N_births

*Keeping only cases where the grandmother had at least two births 
keep if G0_Births>=2

*Drop cases where birth of child is outside possible age ranges for the mother (15-50)
drop if Mother_birthdate==.
gen Motheragebirth=(Child_birthdate-Mother_birthdate)/365.2425
gen temp=(Motheragebirth<15 | Motheragebirth>50)
bysort MotherID: egen outsideRange=max(temp)
qui tab MotherID  if outsideRange==1, nofreq
display "Number of unique mothers with births outside the range 15-50: " r(r)
drop if outsideRange==1
drop temp outsideRange

*Drop cases where birth of child is outside possible age ranges for the grandmother (30-100)
*Cases where the birth date of the grandmother is unknown are not dropped
gen GO_age_birth= (Child_birthdate-Grandmother_birthdate )/365.2425
bysort GrandmotherID: egen min_G0_age_birth=min(GO_age_birth)
bysort GrandmotherID: egen max_G0_age_birth=max(GO_age_birth)
gen temp=(min_G0_age_birth<30 | max_G0_age_birth>100)
replace temp=0 if Grandmother_birthdate==.
bysort GrandmotherID: egen outsideRange=max(temp)
qui tab MotherID  if outsideRange==1, nofreq
display "Number of unique mothers where the minimum age at birth of the grandmother is below 30 or above 100: " r(r)
drop if outsideRange==1
drop min_G0_age_birth GO_age_birth outsideRange temp


*Drop cases where the sex of the child is not male or female (e.g. unknown) 
drop if Child_sex!="Female" & Child_sex!="female" & Child_sex!="Male" & Child_sex!="male" 


/*Descriptive information about birth year of the child, mother and grandmother. 
This information can be used to make further refinements to the study sample 
(e.g. by dropping observations where the grandmother/mother/child is born before 
or after a certain year)*/
sum Child_birthyear
sum Mother_birthyear
sum Grandmother_birthyear

centile Child_birthyear,centile(0(10)100)
centile Mother_birthyear,centile(0(10)100)
centile Grandmother_birthyear,centile(0(10)100)




save Study_sample.dta, replace




/*******************************************************************************
				PART 3: INFANT MORTALITY RATE GRAPH
IMR and the corresponding graph are calculated for the entire population and 
not just for the study sample. The graph can be used to provide some background 
information on the area and period under analysis.						
*******************************************************************************/

use INDIVIDUAL.dta,clear
replace Type = lower(Type)
keep if Type=="birth_date"
gen bdate=mdy(Month,Day,Year)
format bdate %td
drop if bdate==.
keep Id_I bdate
save bdate.dta, replace

use INDIVIDUAL.dta,clear
replace Type = lower(Type)
keep if Type=="death_date"
gen ddate=mdy(Month,Day,Year)
format ddate %td
drop if ddate==.
keep Id_I ddate
merge 1:1 Id_I using bdate.dta
keep if _merge==3
drop _merge
gen dage = (ddate-bdate)/365.2425
keep if dage<1
gen Year=year(ddate)
collapse (count) Num_deaths=Id_I,by(Year)
save numdeaths.dta, replace



use INDIVIDUAL.dta,clear
replace Type = lower(Type)
gen keeprow=1 if Type=="birth_date" & Date_type=="Event"
qui tab Type if keeprow==1
if r(N)==0{
replace keeprow=1 if (Type=="baptism" | Type=="baptism_date") & Date_type=="Event"
keep Id_I keeprow
keep if keeprow==1
merge m:1 Id_I using bdate.dta
keep if _merge==3
drop _merge
gen Year=year(bdate)
}
keep if keeprow==1
collapse (count) Num_births=Id_I,by(Year)
merge 1:1 Year using numdeaths.dta
drop if _merge==2
drop _merge
replace Num_deaths=0 if Num_deaths==.
gen IMR_population=(Num_deaths/Num_births)*1000
keep Year IMR_population
save IMR_population.dta, replace

erase bdate.dta


use Study_sample.dta, clear
keep if Infant_death==1
gen Year=year(date2)
collapse (count) Num_deaths=ChildID,by(Year)
save numdeaths.dta, replace

use Study_sample.dta, clear
keep if date1==Child_birthdate
gen Year=year(date1)
collapse (count) Num_births=ChildID,by(Year)
merge 1:1 Year using numdeaths.dta
drop if _merge==2
drop _merge
replace Num_deaths=0 if Num_deaths==.
gen IMR_sample=(Num_deaths/Num_births)*1000
keep Year IMR_sample
save IMR_sample.dta, replace

erase numdeaths.dta



use IMR_population.dta, clear
merge 1:1 Year using IMR_sample.dta
keep if _merge==3
drop _merge



*Nicer looking graph adjusting the x axis to user's selection
/*Change values of Min_year, Max_year and Axis_spacing
Min and max year are the first and last year of your selected analysis period
Axis spacing is how many years should be between each tick mark of the x axis
*/ 

global Min_year = 1720
global Max_year = 1968
global Axis_spacing=30

if round($Min_year, $Axis_spacing ) >= $Min_year{
global faxis = round($Min_year, $Axis_spacing ) 
}
if round($Min_year, $Axis_spacing ) < $Min_year{
global faxis = round($Min_year, $Axis_spacing ) + $Axis_spacing
}

if round($Max_year, $Axis_spacing) <= $Max_year{
global laxis =round($Max_year, $Axis_spacing)
}
if round($Max_year, $Axis_spacing) > $Max_year{
global laxis =round($Max_year, $Axis_spacing) - $Axis_spacing
}



label var IMR_sample "Study sample"
label var IMR_population "Population"

line IMR_population IMR_sample Year if Year>=$Min_year & Year<=$Max_year , ///
lcolor(black red)  ///
ytick(,nogrid ) ymtick(,nogrid)  lwidth(medium) ///
xmtick($Min_year $faxis ($Axis_spacing) $laxis $Max_year ,nogrid ) ///
xlab($Min_year $faxis ($Axis_spacing) $laxis $Max_year) ///
ylab(,nogrid) ///
graphregion(fcolor(white) lcolor(white) lwidth(vthin)) ///
xtitle("Year", margin(t+1.5)) ytitle("IMR",margin(r+1.5)) ///
plotregion( lcolor(black) lwidth(medium)) ///
legend(region(lcolor(white))) 


graph display, xsize(9.6) ysize(4)
graph export IMR.tif,width(960) replace





/*******************************************************************************
					PART 4: DATA QUALITY CHECKS OF THE STUDY SAMPLE

*******************************************************************************/
use Study_sample.dta, clear

*percentage of births by sex
tab Child_sex if date1== Child_birthdate
*percentage of infant deaths by sex
tab Child_sex if Infant_death==1

*Percentage of infant deaths in the first month of life
stset date2, failure(Infant_death) id(ChildID) time0(date1) origin(time Child_birthdate) 
stsplit month1,at(30)
replace Infant_death = 0 if Infant_death==.
tab month1 if Infant_death==1

sts graph,cumhaz graphregion(fcolor(white) lcolor(white) lwidth(vvthin)) saving(cumhaz_sample, replace) ///
plotregion( lcolor(black) lwidth(medium)) ///
title("Study sample",color(black) margin(t-2) size(medlarge)) xtitle("Days since birth",margin(r+2) size(medlarge))  



*********Compare data quality checks to entire dataset 
use Episodes_file.dta, clear
/*Adjust period of analysis to match the peirod of analysis of the study sample
In this example of code, change 1720 and 1968 to correct years for your study sample:
gen year=year(date1)
drop if year<1720 | year>1968
*/

*percentage of births by sex
tab Child_sex if date1== Child_birthdate
*percentage of infant deaths by sex
tab Child_sex if Infant_death==1

*Percentage of infant deaths in the first month of life
stset date2, failure(Infant_death) id(Id_I) time0(date1) origin(time Child_birthdate) 
stsplit month1,at(30)
replace Infant_death = 0 if Infant_death==.
tab month1 if Infant_death==1

sts graph ,cumhaz graphregion(fcolor(white) lcolor(white) lwidth(vvthin)) saving(cumhaz_data, replace) ///
plotregion( lcolor(black) lwidth(medium)) ///
title("Full dataset",color(black) margin(t-2) size(medlarge)) xtitle("Days since birth",margin(r+2) size(medlarge)) 


graph combine cumhaz_sample.gph cumhaz_data.gph, graphregion(fcolor(white) lcolor(white) lwidth(vvthin)) /// 
plotregion( lcolor(black) lwidth(medium)) ///
saving(cumhaz_firstYear, replace)


graph display, xsize(9.6) ysize(4)
graph export cumhaz_firstYear.tif,width(960) replace








/*******************************************************************************
				PART 5: DESCRIPTIVE STATISTICS 
An Excel file is produced with descriptive statistics of the variables included 
in the basic model	(Descriptive_statistics.xlsx)					
*******************************************************************************/

use Study_sample.dta, clear


putexcel set Descriptive_statistics.xlsx, replace
qui putexcel A1=("Variable") B1=("Percent") C1=("Mean") D1=("SD") E1=("Min") F1=("Max")

gen i=3

foreach varname in Child_sex G0_InfD_cat G0_Births_cat Birth_order_cat Mother_age_cat {
local lnum =i
qui putexcel A`lnum' =("`varname'")
replace i=i+1
gen j = i
levelsof `varname', local(levels) 
foreach l of local levels {
local lnum =j
qui putexcel A`lnum' =("`l'")  
qui replace j=j+1
}
local lnum =i
tab `varname', matcell(freq) 
qui putexcel B`lnum' =matrix(freq/r(N)) 
drop j
qui replace i=i+r(r)+1 
}


foreach varname in Child_birthyear Child_birthdate_cent {
sum `varname'
local lnum =i
qui putexcel A`lnum' =("`varname'") C`lnum' =(r(mean)) D`lnum' =(r(sd)) 
qui putexcel E`lnum' =(r(min)) F`lnum' =(r(max))
qui replace i=i+2 
}

drop i


 
*Tabulations of other variables not included in the basic model
tab G0_InfD 
tab G0_mult_b 
tab G0_Births_unknownBdate 








/*******************************************************************************
						PART 6: SURVIVAL MODELS

Hazard curves are first drawn and saved.
				
After displaying the results of the main models and the sensitivity analysis 
the results are stored in two Excel files (Results_survival.xlsx and 
Results_survival_sensitivity.xlsx)			
*******************************************************************************/

use Study_sample.dta, clear


stset date2, failure(Infant_death) id(ChildID) time0(date1) origin(time Child_birthdate) scale(365.2425)

***Hazard curves
sts graph ,by(G0_InfD_cat) hazard title("Hazard by infant mort of grandmother")
graph export haz_infmorth.tif, replace
sts graph ,by(G0_InfD_cat) cumhaz title("Cumulative hazard by infant mort of grandmother")
graph export cumhaz_infmorth.tif, replace
sts graph ,by(Child_sex) hazard title("Hazard by child sex")
graph export haz_sex.tif, replace
sts graph ,by(Mother_age_cat) hazard title("Hazard by mother's age")
graph export haz_mothage.tif, replace



*Simple Cox model 
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat, efron
stphtest,detail
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat, efron nohr

*Weibull parametric survival model with shared frailty on the mother
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat ,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared
estat ic
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat ,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared nohr

*Weibull parametric survival model with shared frailty on the grandmother
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat ,  ///
shared(GrandmotherID) dist(weibull) frailty(gamma) forceshared
estat ic
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat ,  ///
shared(GrandmotherID) dist(weibull) frailty(gamma) forceshared nohr


*** Save results of the model to Excel 

putexcel set Results_survival.xlsx, replace
putexcel A3=("N. of infant deaths of the grandmother, (ref: 0)") A4=("1 infant death") A5=("2+ infant death") 
putexcel A6=("N. of births of the grandmother (ref:2)") A7=("3 births") A8=("4-6 births") A9=("7+ births")
putexcel A10=("Female") A11=("Male") 
putexcel A12=("Birth order (ref:1)") A13=("2") A14=("3") A15=("4-6") A16=("7+")
putexcel A17=("Child birth date centered") 
putexcel A18=("Mother age 15-24") A19=("25-34 (ref.)") A20=("35-50") A21=("Intercept")
putexcel A22=("Frailty variance")
putexcel A23=("N of children")  A24=("N of infant deaths") 

putexcel B1=("Cox model") B2=("HR") C2=("p-value")
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat, efron
matrix Res=r(table)'
matrix HR=(Res[1..13,1])
putexcel B3=(1) B4=matrix(HR[1..2,1]) B6=(1) B7=matrix(HR[3..5,1]) B10=(1) B11=matrix(HR[6,1]) B12=(1) B13=matrix(HR[7..12,1]) B19=(1) B20=matrix(HR[13,1])
matrix pval=(Res[1..13,4])
putexcel C3=("ref.") C4=matrix(pval[1..2,1]) C6=("ref.") C7=matrix(pval[3..5,1]) C10=("ref.") C11=matrix(pval[6,1]) C12=("ref.") C13=matrix(pval[7..12,1]) C19=("ref.") C20=matrix(pval[13,1])
putexcel B23=(e(N_sub)) B24=(e(N_fail))

putexcel D1=("Weibull model, mother shared frailty.") D2=("HR") E2=("p-value")
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat ,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared
matrix Res=r(table)'
matrix HR=(Res[1..19,1])
putexcel D3=(1) D4=matrix(HR[1..2,1]) D6=(1) D7=matrix(HR[3..5,1]) D10=(1) D11=matrix(HR[6,1]) D12=(1) D13=matrix(HR[7..12,1]) D19=(1) D20=matrix(HR[13..14,1])
matrix pval=(Res[1..16,4])
putexcel E3=("ref.") E4=matrix(pval[1..2,1]) E6=("ref.") E7=matrix(pval[3..5,1]) E10=("ref.") E11=matrix(pval[6,1]) E12=("ref.") E13=matrix(pval[7..12,1]) E19=("ref.") E20=matrix(pval[13..14,1])
putexcel D22=matrix(HR[19,1])
local a=e(p) 
putexcel E22=`a'
putexcel D23=(e(N_sub)) D24=(e(N_fail))

putexcel F1=("Weibull model, grandmother shared frailty.") F2=("HR") F2=("p-value")
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat ,  ///
shared(GrandmotherID) dist(weibull) frailty(gamma) forceshared
matrix Res=r(table)'
matrix HR=(Res[1..19,1])
putexcel F3=(1) F4=matrix(HR[1..2,1]) F6=(1) F7=matrix(HR[3..5,1]) F10=(1) F11=matrix(HR[6,1]) F12=(1) F13=matrix(HR[7..12,1]) F19=(1) F20=matrix(HR[13..14,1])
matrix pval=(Res[1..16,4])
putexcel G3=("ref.") G4=matrix(pval[1..2,1]) G6=("ref.") G7=matrix(pval[3..5,1]) G10=("ref.") G11=matrix(pval[6,1]) G12=("ref.") G13=matrix(pval[7..12,1]) G19=("ref.") G20=matrix(pval[13..14,1])
putexcel F22=matrix(HR[19,1])
local a=e(p) 
putexcel G22=`a'
putexcel F23=(e(N_sub)) F24=(e(N_fail))





***Robustness checks, focusing on Cox models with clustered standard errors
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if Under_observation==1,cluster(MotherID) efron
stphtest,detail
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050==1,cluster(MotherID) efron nohr
 
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050==1,cluster(MotherID) efron
stphtest,detail
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050==1,cluster(MotherID) efron nohr

xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050death==1, cluster(MotherID)  efron
stphtest,detail
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050death==1, cluster(MotherID)  efron nohr

xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050husband==1, cluster(MotherID) efron
stphtest,detail
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050husband==1, cluster(MotherID) efron nohr

xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050==1 &  G0_Births_unknownBdate==0, cluster(MotherID) efron
stphtest,detail
xi: stcox i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050==1 &  G0_Births_unknownBdate==0, cluster(MotherID) efron nohr





*** Robustness checks 

putexcel set Results_survival_sensitivity.xlsx, replace
putexcel A3=("N. of infant deaths of the grandmother, (ref: 0)") A4=("1 infant death") A5=("2+ infant death") 
putexcel A6=("N. of births of the grandmother (ref:2)") A7=("3 births") A8=("4-6 births") A9=("7+ births")
putexcel A10=("Female") A11=("Male") 
putexcel A12=("Birth order (ref:1)") A13=("2") A14=("3") A15=("4-6") A16=("7+")
putexcel A17=("Child birth date centered") 
putexcel A18=("Mother age 15-24") A19=("25-34 (ref.)") A20=("35-50") A21=("Intercept")
putexcel A22=("Frailty variance")
putexcel A23=("N of children")  A24=("N of infant deaths") 


putexcel B1=("Model 1") B2=("HR") C2=("p-value")
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if Under_observation==1,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared
matrix Res=r(table)'
matrix HR=(Res[1..19,1])
putexcel B3=(1) B4=matrix(HR[1..2,1]) B6=(1) B7=matrix(HR[3..5,1]) B10=(1) B11=matrix(HR[6,1]) B12=(1) B13=matrix(HR[7..12,1]) B19=(1) B20=matrix(HR[13..14,1])
matrix pval=(Res[1..16,4])
putexcel C3=("ref.") C4=matrix(pval[1..2,1]) C6=("ref.") C7=matrix(pval[3..5,1]) C10=("ref.") C11=matrix(pval[6,1]) C12=("ref.") C13=matrix(pval[7..12,1]) C19=("ref.") C20=matrix(pval[13..14,1])
putexcel B22=matrix(HR[19,1])
local a=e(p) 
putexcel C22=`a'
putexcel B23=(e(N_sub)) B24=(e(N_fail))


putexcel D1=("Model 2") D2=("HR") E2=("p-value")
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050==1,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared
matrix Res=r(table)'
matrix HR=(Res[1..19,1])
putexcel D3=(1) D4=matrix(HR[1..2,1]) D6=(1) D7=matrix(HR[3..5,1]) D10=(1) D11=matrix(HR[6,1]) D12=(1) D13=matrix(HR[7..12,1]) D19=(1) D20=matrix(HR[13..14,1])
matrix pval=(Res[1..16,4])
putexcel E3=("ref.") E4=matrix(pval[1..2,1]) E6=("ref.") E7=matrix(pval[3..5,1]) E10=("ref.") E11=matrix(pval[6,1]) E12=("ref.") E13=matrix(pval[7..12,1]) E19=("ref.") E20=matrix(pval[13..14,1])
putexcel D22=matrix(HR[19,1])
local a=e(p) 
putexcel E22=`a'
putexcel D23=(e(N_sub)) D24=(e(N_fail))


putexcel F1=("Model 3") F2=("HR") G2=("p-value")
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050death==1,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared
matrix Res=r(table)'
matrix HR=(Res[1..19,1])
putexcel F3=(1) F4=matrix(HR[1..2,1]) F6=(1) F7=matrix(HR[3..5,1]) F10=(1) F11=matrix(HR[6,1]) F12=(1) F13=matrix(HR[7..12,1]) F19=(1) F20=matrix(HR[13..14,1])
matrix pval=(Res[1..16,4])
putexcel G3=("ref.") G4=matrix(pval[1..2,1]) G6=("ref.") G7=matrix(pval[3..5,1]) G10=("ref.") G11=matrix(pval[6,1]) G12=("ref.") G13=matrix(pval[7..12,1]) G19=("ref.") G20=matrix(pval[13..14,1])
putexcel F22=matrix(HR[19,1])
local a=e(p) 
putexcel G22=`a'
putexcel F23=(e(N_sub)) F24=(e(N_fail))


putexcel H1=("Model 4") H2=("HR") I2=("p-value")
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050husband==1,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared
matrix Res=r(table)'
matrix HR=(Res[1..19,1])
putexcel H3=(1) H4=matrix(HR[1..2,1]) H6=(1) H7=matrix(HR[3..5,1]) H10=(1) H11=matrix(HR[6,1]) H12=(1) H13=matrix(HR[7..12,1]) H19=(1) H20=matrix(HR[13..14,1])
matrix pval=(Res[1..16,4])
putexcel I3=("ref.") I4=matrix(pval[1..2,1]) I6=("ref.") I7=matrix(pval[3..5,1]) I10=("ref.") I11=matrix(pval[6,1]) I12=("ref.") I13=matrix(pval[7..12,1]) I19=("ref.") I20=matrix(pval[13..14,1])
putexcel H22=matrix(HR[19,1])
local a=e(p) 
putexcel I22=`a'
putexcel H23=(e(N_sub)) H24=(e(N_fail))


putexcel J1=("Model 5") J2=("HR") K2=("p-value")
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050==1 &  G0_Births_unknownBdate==0,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared
matrix Res=r(table)'
matrix HR=(Res[1..19,1])
putexcel J3=(1) J4=matrix(HR[1..2,1]) J6=(1) J7=matrix(HR[3..5,1]) J10=(1) J11=matrix(HR[6,1]) J12=(1) J13=matrix(HR[7..12,1]) J19=(1) J20=matrix(HR[13..14,1])
matrix pval=(Res[1..16,4])
putexcel K3=("ref.") K4=matrix(pval[1..2,1]) K6=("ref.") K7=matrix(pval[3..5,1]) K10=("ref.") K11=matrix(pval[6,1]) K12=("ref.") K13=matrix(pval[7..12,1]) K19=("ref.") K20=matrix(pval[13..14,1])
putexcel J22=matrix(HR[19,1])
local a=e(p) 
putexcel K22=`a'
putexcel J23=(e(N_sub)) J24=(e(N_fail))


putexcel L1=("Model 6") L2=("HR") L2=("p-value")
xi: streg  i.G0_InfD_cat i.G0_Births_cat i.Child_sex i.Birth_order_cat Child_birthdate_cent i.Mother_age_cat if G0_obs_2050==1 &  Under_observation==1,  ///
shared(MotherID) dist(weibull) frailty(gamma) forceshared
matrix Res=r(table)'
matrix HR=(Res[1..19,1])
putexcel L3=(1) L4=matrix(HR[1..2,1]) L6=(1) L7=matrix(HR[3..5,1]) L10=(1) L11=matrix(HR[6,1]) L12=(1) L13=matrix(HR[7..12,1]) L19=(1) L20=matrix(HR[13..14,1])
matrix pval=(Res[1..16,4])
putexcel M3=("ref.") M4=matrix(pval[1..2,1]) M6=("ref.") M7=matrix(pval[3..5,1]) M10=("ref.") M11=matrix(pval[6,1]) M12=("ref.") M13=matrix(pval[7..12,1]) M19=("ref.") M20=matrix(pval[13..14,1])
putexcel L22=matrix(HR[19,1])
local a=e(p) 
putexcel M22=`a'
putexcel L23=(e(N_sub)) L24=(e(N_fail))

